/**
  ******************************************************************************
  * @file    py32f410_hal_i2c_ex.c
  * @author  MCU Application Team
  * @brief   I2C Extended HAL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of I2C Extended peripheral:
  *           + Extended features functions
  *
  @verbatim
  ==============================================================================
               ##### I2C peripheral Extended features  #####
  ==============================================================================

  [..] Comparing to other previous devices, the I2C interface for PY32F410
       devices contains the following additional features

       (+) Possibility to disable or enable Analog Noise Filter
       (+) Use of a configured Digital Noise Filter
       (+) Disable or enable wakeup from Stop mode(s)
       (+) Disable or enable Fast Mode Plus

                     ##### How to use this driver #####
  ==============================================================================
  [..] This driver provides functions to configure Noise Filter and Wake Up Feature
    (#) Configure I2C Analog noise filter using the function HAL_I2CEx_ConfigAnalogFilter()
    (#) Configure I2C Digital noise filter using the function HAL_I2CEx_ConfigDigitalFilter()
    (#) Configure the enable or disable of I2C Wake Up Mode using the functions :
          (++) HAL_I2CEx_EnableWakeUp()
          (++) HAL_I2CEx_DisableWakeUp()
    (#) Configure the enable or disable of fast mode plus driving capability using the functions :
          (++) HAL_I2CEx_EnableFastModePlus()
          (++) HAL_I2CEx_DisableFastModePlus()
  @endverbatim
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "py32f4xx_hal.h"

/** @addtogroup PY32F410_HAL_Driver
  * @{
  */

/** @defgroup I2CEx I2CEx
  * @brief I2C Extended HAL module driver
  * @{
  */

#ifdef HAL_I2C_MODULE_ENABLED

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/** @defgroup I2CEx_Exported_Functions I2C Extended Exported Functions
  * @{
  */

/** @defgroup I2CEx_Exported_Functions_Group1 Extended features functions
  * @brief    Extended features functions
  *
@verbatim
 ===============================================================================
                      ##### Extended features functions #####
 ===============================================================================
    [..] This section provides functions allowing to:
      (+) Configure Noise Filters
      (+) Configure Wake Up Feature
      (+) Configure Fast Mode Plus

@endverbatim
  * @{
  */

/**
  * @brief  Enable I2C wakeup from Stop mode(s).
  * @param  hi2c Pointer to a I2C_HandleTypeDef structure that contains
  *                the configuration information for the specified I2Cx peripheral.
  * @param  ToutDiv specifies the clock division frequency used to count Wake up timeout times.
  *          This parameter can be one of the following values:
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CLK_DIV_128  128 division applied to Wake up timeout clock
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CLK_DIV_256  256 division applied to Wake up timeout clock
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CLK_DIV_512  512 division applied to Wake up timeout clock
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CLK_DIV_1024 1024 division applied to Wake up timeout clock
  * @param  ToutCnt specifies the clock count value of the wake up timeout 
  *          This parameter can be one of the following values:
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CNT_2   clock count value of the wake up timeout is 2
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CNT_8   clock count value of the wake up timeout is 8
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CNT_32  clock count value of the wake up timeout is 32
  *            @arg @ref I2C_WAKEUP_TIMEOUT_CNT_128 clock count value of the wake up timeout is 128
  * @retval HAL status
  */
HAL_StatusTypeDef HAL_I2CEx_EnableWakeUp(I2C_HandleTypeDef *hi2c,uint32_t ToutDiv,uint32_t ToutCnt)
{
  /* Check the parameters */
  assert_param(IS_I2C_WAKEUP_FROMSTOP_INSTANCE(hi2c->Instance));
  assert_param(IS_I2C_WAKEUP_TIMEOUT_CLK_DIV(ToutDiv));
  assert_param(IS_I2C_WAKEUP_TIMEOUT_CNT(ToutCnt));
  
  if (hi2c->State == HAL_I2C_STATE_READY)
  {
    /* Process Locked */
    __HAL_LOCK(hi2c);

    hi2c->State = HAL_I2C_STATE_BUSY;

    /* Disable the selected I2C peripheral */
    __HAL_I2C_DISABLE(hi2c);

    /* Enable wakeup from stop mode */
    hi2c->Instance->CR1 |= I2C_CR1_WUPEN;

    /* Configure clock division and clock count value of the wake up timeout */
    MODIFY_REG(hi2c->Instance->CR1, (I2C_CR1_WKUP_DIV | I2C_CR1_WKUP_CNT), (ToutDiv | ToutCnt));

    __HAL_I2C_ENABLE(hi2c);

    hi2c->State = HAL_I2C_STATE_READY;

    /* Process Unlocked */
    __HAL_UNLOCK(hi2c);

    return HAL_OK;
  }
  else
  {
    return HAL_BUSY;
  }
}

/**
  * @brief  Disable I2C wakeup from Stop mode(s).
  * @param  hi2c Pointer to a I2C_HandleTypeDef structure that contains
  *                the configuration information for the specified I2Cx peripheral.
  * @retval HAL status
  */
HAL_StatusTypeDef HAL_I2CEx_DisableWakeUp(I2C_HandleTypeDef *hi2c)
{
  /* Check the parameters */
  assert_param(IS_I2C_WAKEUP_FROMSTOP_INSTANCE(hi2c->Instance));

  if (hi2c->State == HAL_I2C_STATE_READY)
  {
    /* Process Locked */
    __HAL_LOCK(hi2c);

    hi2c->State = HAL_I2C_STATE_BUSY;

    /* Disable the selected I2C peripheral */
    __HAL_I2C_DISABLE(hi2c);

    /* Enable wakeup from stop mode */
    hi2c->Instance->CR1 &= ~(I2C_CR1_WUPEN);

    __HAL_I2C_ENABLE(hi2c);

    hi2c->State = HAL_I2C_STATE_READY;

    /* Process Unlocked */
    __HAL_UNLOCK(hi2c);

    return HAL_OK;
  }
  else
  {
    return HAL_BUSY;
  }
}

/**
  * @brief Enable the I2C fast mode plus driving capability.
  * @param ConfigFastModePlus Selects the pin.
  *   This parameter can be one of the @ref I2CEx_FastModePlus values
  * @note  For I2C1, fast mode plus driving capability can be enabled on all selected
  *        I2C1 pins using I2C_FASTMODEPLUS_I2C1 parameter or independently
  *        on each one of the following pins PA4,PA9,PA10,PA13,PB4,PB6,PB7,PB8,PB9,PC0,
  *        PC1,PC6,PC7,PC9,PD4,PD5.
  * @note  For I2C2, fast mode plus driving capability can be enabled on all selected
  *        I2C2 pins using I2C_FASTMODEPLUS_I2C2 parameter or independently
  *        on each one of the following pins PA1,PA3,PA7,PA8,PA11,PA12,PB10,PB11,PB13,
  *        PB14,PC0,PC1,PC9,PD6,PD7.
  * @retval None
  */
void HAL_I2CEx_EnableFastModePlus(uint32_t ConfigFastModePlus)
{
  /* Check the parameter */
  assert_param(IS_I2C_FASTMODEPLUS(ConfigFastModePlus));

  /* Enable SYSCFG clock */
  __HAL_RCC_SYSCFG_CLK_ENABLE();

  /* Enable fast mode plus driving capability for selected pin */
  SET_BIT(SYSCFG->CFGR[0], (uint32_t)ConfigFastModePlus);
}

/**
  * @brief Disable the I2C fast mode plus driving capability.
  * @param ConfigFastModePlus Selects the pin.
  *   This parameter can be one of the @ref I2CEx_FastModePlus values
  * @note  For I2C1, fast mode plus driving capability can be enabled on all selected
  *        I2C1 pins using I2C_FASTMODEPLUS_I2C1 parameter or independently
  *        on each one of the following pins PA4,PA9,PA10,PA13,PB4,PB6,PB7,PB8,PB9,PC0,
  *        PC1,PC6,PC7,PC9,PD4,PD5.
  * @note  For I2C2, fast mode plus driving capability can be enabled on all selected
  *        I2C2 pins using I2C_FASTMODEPLUS_I2C2 parameter or independently
  *        on each one of the following pins PA1,PA3,PA7,PA8,PA11,PA12,PB10,PB11,PB13,
  *        PB14,PC0,PC1,PC9,PD6,PD7.
  * @retval None
  */
void HAL_I2CEx_DisableFastModePlus(uint32_t ConfigFastModePlus)
{
  /* Check the parameter */
  assert_param(IS_I2C_FASTMODEPLUS(ConfigFastModePlus));

  /* Enable SYSCFG clock */
  __HAL_RCC_SYSCFG_CLK_ENABLE();

  /* Disable fast mode plus driving capability for selected pin */
  CLEAR_BIT(SYSCFG->CFGR[0], (uint32_t)ConfigFastModePlus);
}

/**
  * @}
  */

/**
  * @}
  */

#endif /* HAL_I2C_MODULE_ENABLED */
/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
